/*	
 Copyright 2014 Adobe Systems Incorporated.  All rights reserved. 

Purpose- 
This file has the implementation of infrastructure needed for liveView extensions
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global DW_EVENTS, DW_LIVEEDIT_CONSTANTS, CustomEvent, DW_EXTENSION_EVENT, Node */

/*
DWObject - exposes functions which need to access DW (like querying and updating document)
*/
(function () {
    'use strict';
    
    function DWObjectForExtensions() {
    }

    DWObjectForExtensions.prototype.getAppLanguage = function () {
        return window.appLanguage;
    };
    
    /*
    Function: getDWDocumentClassesAndIds - Gets the list of available classes and Ids for use in this document

    Argument: callbackfunc -> callback to be called after executing

    Return Object : 
    allSelectors - array of available selectors
    */
    DWObjectForExtensions.prototype.getDWDocumentClassesAndIds = function (callbackfunc) {
        var callbackObj = {};
        callbackObj.callback = callbackfunc;
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'getDWDocumentClassesAndIds', callbackObj, false);
    };
    
    /*
    Function: getDWDocumentElementClassAndID - Gets the class and id attributes from document for the passend element

    Argument:   elem - Corresponding live view element
                callbackfunc - callback to be called after executing

    Return Object : object.classStr - Value of class attribute
                    object.idStr    - Value of id attribute
    */
    DWObjectForExtensions.prototype.getDWDocumentElementClassAndID = function (elem, callbackfunc) {
        if (!elem) {
            return;
        }
        
        //get the unique ID for the element to pass to DW side
        var uniqId = elem.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
        
        var argObj = {};
        argObj.uniqId = uniqId;
        argObj.callback = callbackfunc;
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'getDWDocumentElementClassAndID', argObj, false);
    };

    /*
    function:updateDWDocElementAttr - update an attribute value in the DW document for the element

    Arguments:

        elem - Corresponding live view element
        attrName - what attribute needs to be changed
        attrValue - new value of the attribute

    Return Obj : Nothing
    */
    DWObjectForExtensions.prototype.updateDWDocumentElementAttr = function (elem, attrName, attrVal, callback, selId) {
        if (!elem) {
            return;
        }

        //get the unique ID for the element to pass to DW side
        var uniqId = elem.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
        if (!uniqId || uniqId.length === 0) {
            return;
        }

        //propogate to DW
        var argObj = {};
        argObj.uniqId = uniqId;
        argObj.attrName	= attrName;
        argObj.attrValue = attrVal;
        if (callback) {
            argObj.callback = callback;
        }
		if (selId) {
			argObj.selectionId = selId;
		}
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'updateDWDocElementAttr', argObj, true);
    };
    DWObjectForExtensions.prototype.sendNFWUpdateToBridge = function (e, c) {
        if (!e) {
            return;
        }
       
        var argObj = {};
        argObj.type = e;
        argObj.category = c;
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'sendNFWUpdate', argObj, false);
    };
    DWObjectForExtensions.prototype.sendTextHudOpenUpdateforNFW = function (argObj) {
        if (!argObj) {
            return;
        }
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'sendTextHudOpenUpdateforNFW', argObj, false);
    };
    /**
     * function:dwBrowseForFileURL
     * (Internal function on spider monkey side)
     * @param argObj
     * operation: select/open
     * subop : name that will be shown at the top
     * fileter: filters to be applied while showing the folder 
     * callback: on successful/failure of execution callback to js layer.
     * Description: 
     * Browse for file functionality for source and link.
     */
    DWObjectForExtensions.prototype.dwBrowseForFileURL = function (operation, subOperation, callback, filter, stopEditOp) {
        if (!operation || !subOperation) {
            return;
        }

        //propogate to DW
        var argObj = {};
        argObj.filter = filter;
        argObj.operation = operation;
        argObj.subOp = subOperation;
        argObj.callback = callback;

        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, "dwBrowseForFileURL", argObj, stopEditOp);
    };
    
    /*
    Function: logHeadlightsData
    Arguments: 
    category -> feature area in which this event occurred
    eventString -> event occurred
    */
    
    DWObjectForExtensions.prototype.logHeadlightsData = function (category, eventString) {
        var argObj = {};
        argObj.category = category;
        argObj.eventString = eventString;
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, "DWLogHeadlightsData", argObj, false);
    };
    /*
    function:DWSMGetDWIdForAnchorTag
    (Internal function on spider monkey side)
    Arguments:argObj
        
        imgId - dwid of image under edit
        parentId - dwid ofParentNode of that image
        newText    - new text to keep in the pplace of old content at that offset.
        counter - how many nodes have been modified(to get corresponding node dwid's for tempid's)
        callback - callback function to be called
    
    Description: 
        Update the link attribute and fetch dwid's for all the modfied nodes.
    
    */
    DWObjectForExtensions.prototype.DWSMGetDWIdForAnchorTag = function (argObj) {
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, "DWSMGetDWIdForAnchorTag", argObj, true);
    };
    
    /*
    Live view object - Used for the loading and handling of extensions in LiveView
    */
    function LiveViewObjectForExtensions() {
        this.currentSelectedElement = null;
        this.isANonEditableElement = false;
        this.liveViewExtensionsArray = [];
		// Hard coding the help ids for now. Will change it once we have the proper mapping.
		this.liveViewExtensionsArrayForHelp = [{extensionId: "dwImageHud", helpId: "AH_IMAGE_HUD"}, {extensionId: "editableSelectors", helpId: "AH_ESH"}];
        this.auxiliaryIframeExists = {};
        this.dwObject = new DWObjectForExtensions();
		this.liveExtensionsConfigPath = null;
        this.auxHudVisbility = {};
    }

    /*
    function:getCurrentSelectedElement - get the current selected node in Live View. This is the live view 
                                         element corresponding to the selected element in DW DOM.

    Arguments: none

    Return: Returns the DOM element which is set
            using setCurrentSelectedElement function
    */
    LiveViewObjectForExtensions.prototype.getCurrentSelectedElement = function () {
        return this.currentSelectedElement;
    };

    LiveViewObjectForExtensions.prototype.setAuxHudVisibility = function (tag, extensionID, visibility) {
        if (!this.auxHudVisbility.hasOwnProperty(tag)) {
            this.auxHudVisbility[tag] = {};
        }
        
        this.auxHudVisbility[tag][extensionID] = visibility;
    };
    
    LiveViewObjectForExtensions.prototype.getAuxHudVisibility = function (tag, visibility) {
        if (this.auxHudVisbility.hasOwnProperty(tag)) {
            var auxHudVisibile = false,
                property;
            for (property in this.auxHudVisbility[tag]) {
                if (this.auxHudVisbility[tag].hasOwnProperty(property)) {
                    auxHudVisibile = auxHudVisibile || this.auxHudVisbility[tag][property];
                }
            }
            return auxHudVisibile;
        }
        
        return false;
    };
    
    /*
    function:setCurrentSelectedElement - set the current selected node in Live View. This is the live view 
                                         element corresponding to the selected element in DW DOM.

    Arguments: curElement - DOM element which is to be set as the current selection

    Return: none
    */
    LiveViewObjectForExtensions.prototype.setCurrentSelectedElement = function (curElement, elementIsNonEditable) {
        this.currentSelectedElement = curElement;
        if (elementIsNonEditable === 'true') {
            this.isANonEditableElement = true;
        } else {
            //we need to check whether the element is in editable region in case of template instance
            this.isANonEditableElement = window.liveEditTemplateInstance ? !this.isElementInsideEditableRegion(curElement) : false;
        }
    };
    
    /*
    function:setSelectionInTemplateInstance - Function to set the selection in a Template Instance Page. Sets the passed element as
                                                our selected element and sets whether it is editable based on whether the element is
                                                is inside Editable Region or not.
    Arguments: curElement - DOM element which is to be set as the current selection

    Return: none
    */
    LiveViewObjectForExtensions.prototype.setSelectionInTemplateInstance = function (curElement) {
        var isNonEditable = !this.isElementInsideEditableRegion(curElement);
        //editable elements in template instances are selected through selection change messages
        //non editable regions cannot be selected in DW, so we have to set selection on them through
        //clicks in Live View
        if (isNonEditable) {
            this.setCurrentSelectedElement(curElement, 'true');
            //send a selection changed event to notify extensions
            var dwSelectionEvent = {};
            dwSelectionEvent.type = DW_EVENTS.SelectionChanged;
            this.messageToExtensions(dwSelectionEvent);
        }
    };

    /*
    function:isElementEditable - Check whether the passed element is editable from live view.
                                 check is based on the presence of dw node ID

    Arguments: curElement - DOM element

    Return: true if editable, false otherwise
    */
    LiveViewObjectForExtensions.prototype.isElementEditable = function (element) {
        var editable = false;
        if (element && !this.isANonEditableElement) {
            var tagUniqId = element.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
            if (tagUniqId && tagUniqId.length > 0) {
                editable = true;
            }
        }

        return editable;
    };
    
    /*
    function:isElementInsideEditableRegion - Checks if the passed element is inside DW Template 
                                             Editable Region bounds

    Arguments: element

    Return: true if the element is inside an Editable region, false otherwise
    */
    LiveViewObjectForExtensions.prototype.isElementInsideEditableRegion = function (element) {
        if (!element) {
            return false;
        }

        var isInsideEditableRegion = false;
        var curNode = element;
        while (curNode) {
            //DW templates uses comments for marking regions
            if (curNode.nodeType === Node.COMMENT_NODE) {
                // if we reach ER End first, then element is outside of ER.
                if (curNode.nodeValue.indexOf(DW_LIVEEDIT_CONSTANTS.EditableRegionEndText) === 0) {
                    break;
                }

                // if we reach ER Begin first, then element is inside ER.
                if (curNode.nodeValue.indexOf(DW_LIVEEDIT_CONSTANTS.EditableRegionBeginText) === 0) {
                    isInsideEditableRegion = true;
                    break;
                }
            }

            // If no previous sibling, then take the parent and search
            if (curNode.previousSibling) {
                curNode = curNode.previousSibling;
            } else {
                curNode = curNode.parentNode;
            }
        }
        
        return isInsideEditableRegion;
    };

    /*
    function:getElementRect - Returns the containing rect for the element in the document

    Arguments: elem - DOM element

    Return: containing rect(top, left, width, height) of the element
    */
    LiveViewObjectForExtensions.prototype.getElementRect = function (elem) {
        var rect = { top : 0, left : 0, width : 0, height : 0 };
        if (elem) {
            var boudingRect = elem.getBoundingClientRect();
            rect.top = boudingRect.top + window.scrollY;
            rect.left = boudingRect.left + window.scrollX;
            rect.width = boudingRect.right - boudingRect.left;
            rect.height = boudingRect.bottom - boudingRect.top;
        }
        return rect;
    };
    
    /*
    function:clearBrowserUndoRedos - Clears browser Undo/Redo Stack. Undo/Redo goes to Titandoc only when 
                                     browser undo/redo stack is empty.
    Arguments: none

    Return: none
    */
    LiveViewObjectForExtensions.prototype.clearBrowserUndoRedos = function () {
        //clear the stack with the new command
        document.execCommand("ClearUndoRedoOperations", false, null);

        //DW updates the Live View Undo/Redo status on selection change
        //so initiate a selection change
        var selectionChangeEvent = new CustomEvent("selectionchange");
        document.dispatchEvent(selectionChangeEvent);
    };

    /*
    function:showExtensionById - display the extension

    Arguments: extensionID - id of the extension

    Return: none
    */
    LiveViewObjectForExtensions.prototype.showExtensionById = function (extensionID) {
        var extensionIndex = this.getExtensionIndexById(extensionID);
        if (extensionIndex >= 0) {
            //show only if enabled
            if (this.liveViewExtensionsArray[extensionIndex].enabled === true) {
                var extensionIFrame = this.liveViewExtensionsArray[extensionIndex].frame;
                if (extensionIFrame && extensionIFrame.style) {
                    extensionIFrame.style.display = 'block';
                }
            }
        }
    };

    /*
    function:hideExtensionById - hide the extension

    Arguments: extensionID - id of the extension

    Return: none
    */
    LiveViewObjectForExtensions.prototype.hideExtensionById = function (extensionID) {
        var extensionIFrame = this.getExtensionIFrameById(extensionID);
        if (extensionIFrame && extensionIFrame.style) {
            extensionIFrame.style.display = 'none';
        }
    };
    
    /*
    function:isExtensionVisible - return the visibility of extension by ID

    Arguments: extensionID - id of the extension

    Return: true if visible, false otherwise
    */
    LiveViewObjectForExtensions.prototype.isExtensionVisible = function (extensionID) {
        var isVisible = false;
        var extensionIFrame = this.getExtensionIFrameById(extensionID);
        if (extensionIFrame && extensionIFrame.style) {
            isVisible = (extensionIFrame.style.display !== 'none');
        }
        
        return isVisible;
    };

    /*
    function:disableAllExtensions - disable all extensions

    Arguments: none

    Return: none
    */
    LiveViewObjectForExtensions.prototype.disableAllExtensions = function () {
        var i;
        for (i = 0; i < this.liveViewExtensionsArray.length; i++) {
            if (this.liveViewExtensionsArray[i]) {
                var extensionIFrame = this.liveViewExtensionsArray[i].frame;
                this.liveViewExtensionsArray[i].enabled = false;
                if (extensionIFrame && extensionIFrame.style) {
                    extensionIFrame.style.display = 'none';
                }
            }
        }
    };

    /*
    function:enableAllExtensions - enable all extensions

    Arguments: none

    Return: none
    */
    LiveViewObjectForExtensions.prototype.enableAllExtensions = function () {
        var i;
        for (i = 0; i < this.liveViewExtensionsArray.length; i++) {
            if (this.liveViewExtensionsArray[i]) {
                this.liveViewExtensionsArray[i].enabled = true;
            }
        }
    };

    /*
    function:positionExtensionById - position an extension in the document

    Arguments: extension ID, top, left, width and height to be set

    Return: none
    */
    LiveViewObjectForExtensions.prototype.positionExtensionById = function (extensionID, top, left, width, height) {
        var extensionIndex = this.getExtensionIndexById(extensionID);
        if (extensionIndex >= 0) {
            //position only if enabled
            if (this.liveViewExtensionsArray[extensionIndex].enabled === true) {
                var extensionIFrame = this.liveViewExtensionsArray[extensionIndex].frame;
                //if document body or html element is positioned relative
                //then their top left may not be (0,0). We should reduce this
                //in our computations
                var topOffset = 0, leftOffset = 0, documentHtmlStyle = null;
                var documentBodyStyle = window.getComputedStyle(document.body);
                var documentHTMLElement = document.getElementsByTagName("html")[0];
                if (documentHTMLElement) {
                    documentHtmlStyle = window.getComputedStyle(documentHTMLElement);
                }
                if (documentBodyStyle.position === "relative") {
                    var bodyRect = this.getElementRect(document.body);
                    topOffset = bodyRect.top;
                    leftOffset = bodyRect.left;
                } else if (documentHtmlStyle && documentHtmlStyle.position === "relative") {
                    var htmlRect = this.getElementRect(documentHTMLElement);
                    topOffset = htmlRect.top;
                    leftOffset = htmlRect.left;
                }
                
                if (extensionIFrame && extensionIFrame.style) {
                    extensionIFrame.style.top = (top - topOffset) + "px";
                    extensionIFrame.style.left = (left - leftOffset) + "px";
                    extensionIFrame.style.height = height + "px";
                    extensionIFrame.style.width = width + "px";
                    extensionIFrame.style.display = "block";
                }
            }
        }
    };

    /*
    function:getExtensionIFrameById - get the iframe for the extension in the document

    Arguments: extension ID

    Return: IFrame element for the extention if found
    */
    LiveViewObjectForExtensions.prototype.getExtensionIFrameById = function (extensionID) {
        var extensionIFrame = null;
        var extensionIndex = this.getExtensionIndexById(extensionID);
        if (extensionIndex >= 0) {
            extensionIFrame = this.liveViewExtensionsArray[extensionIndex].frame;
        }
        return extensionIFrame;
    };

    /*
    function:getExtensionIndexById - get the index for the extension in the list

    Arguments: extension ID

    Return: index of the extension if found, -1 otherwise
    */
    LiveViewObjectForExtensions.prototype.getExtensionIndexById = function (extensionID) {
        var extensionIndex = -1;
        var i;
        if (extensionID && extensionID.length > 0) {
            for (i = 0; i < this.liveViewExtensionsArray.length; i++) {
                if (this.liveViewExtensionsArray[i] && this.liveViewExtensionsArray[i].id === extensionID) {
                    extensionIndex = i;
                    break;
                }
            }
        }
        return extensionIndex;
    };

    /*
    function:loadAllExtensions - load each extension to an iframe and add it to document

    Arguments: none

    Return: none
    */
    LiveViewObjectForExtensions.prototype.loadAllExtensions = function () {
        var i;
      
        //make sure we have the details of extensions preloaded
        if (!window.extensionsDetail || !window.extensionsResourcePath) {
            return;
        }

        if (window.location && window.location.protocol !== "file:") {
            //If it is not a file: URL, browser will not load file: URLs in iFrames due to cross origin policy.
            //So our extensions will not load in this case. Here we are faking that our extensions
            //are present in the remote host. We will provide these files from DW CEF resource loader
            ///when we get a load request. SO we need to identify our extension files in the URL.
            //Adding a placeholder here to identify our extension resources. DW CEF resource loader 
            //will look for this placeholder and replace as needed if found. And the browser will think
            //that this is coming from remote host, thus bypassing cross origin policy
            this.liveExtensionsConfigPath = window.location.protocol + "//" + window.location.hostname + "//";
            this.liveExtensionsConfigPath += DW_LIVEEDIT_CONSTANTS.ExtensionsResourcePathPlaceHolder;
        } else {
            this.liveExtensionsConfigPath = window.extensionsResourcePath;
        }
        
        var extensionsDetails = window.extensionsDetail;
        var iFrameCSS = "animation:none;animation-delay:0;animation-direction:normal;animation-duration:0;animation-fill-mode:none;animation-iteration-count:1;animation-name:none;animation-play-state:paused;animation-timing-function:ease;backface-visibility:hidden;background:0;background-attachment:scroll;background-clip:border-box;background-color:transparent;background-image:none;background-origin:padding-box;background-position:00;background-position-x:0;background-position-y:0;background-repeat:repeat;background-size:0;border:0;border-style:none;border-width:medium;border-color:transparent;border-bottom:0;border-bottom-color:transparent;border-bottom-left-radius:0;border-bottom-right-radius:0;border-bottom-style:none;border-bottom-width:medium;border-collapse:separate;border-image:none;border-left:0;border-left-color:transparent;border-left-style:none;border-left-width:medium;border-radius:0;border-right:0;border-right-color:transparent;border-right-style:none;border-right-width:medium;border-spacing:0;border-top:0;border-top-color:transparent;border-top-left-radius:0;border-top-right-radius:0;border-top-style:none;border-top-width:medium;bottom:0;box-shadow:none;box-sizing:content-box;caption-side:top;clear:none;clip:auto;color:#000000;columns:auto;column-count:auto;column-fill:balance;column-gap:normal;column-rule:mediumnonecurrentColor;column-rule-color:currentColor;column-rule-style:none;column-rule-width:none;column-span:1;column-width:auto;content:normal;counter-increment:none;counter-reset:none;cursor:auto;direction:ltr;display:none;empty-cells:show;float:none;font:normal;font-family:tahoma;font-size:medium;font-style:normal;font-variant:normal;font-weight:normal;height:auto;hyphens:none;left:0px;letter-spacing:normal;line-height:normal;list-style:none;list-style-image:none;list-style-position:outside;list-style-type:disc;margin:0;margin-bottom:0;margin-left:0;margin-right:0;margin-top:0;max-height:none;max-width:none;min-height:0;min-width:0;opacity:1;orphans:0;outline:0;outline-color:auto;outline-style:auto;outline-width:auto;overflow:visible;overflow-x:visible;overflow-y:visible;padding:0;padding-bottom:0;padding-left:0;padding-right:0;padding-top:0;page-break-after:auto;page-break-before:auto;page-break-inside:auto;perspective:none;perspective-origin:50%50%;position:absolute;right:auto;tab-size:8;table-layout:auto;text-align:left;text-align-last:auto;text-decoration:none;text-decoration-color:transparent;text-decoration-line:none;text-decoration-style:solid;text-indent:0;text-shadow:none;text-transform:none;top:0px;transform:none;transform-style:flat;transition:none;transition-delay:0s;transition-duration:0s;transition-property:none;transition-timing-function:ease;unicode-bidi:normal;vertical-align:baseline;visibility:visible;white-space:normal;widows:0;width:auto;word-spacing:normal;z-index:99999999999999999999999999;";
        //create extensions container if not present
        var extensionContainer = document.getElementsByTagName(DW_LIVEEDIT_CONSTANTS.GlobalContainerDiv)[0];
        if (!extensionContainer) {
            extensionContainer = document.createElement(DW_LIVEEDIT_CONSTANTS.GlobalContainerDiv);
            //set position property to default value to override * selector
            var containerCSS = "position:static;";
            extensionContainer.style.cssText = containerCSS;
        }

        for (i = 0; i < extensionsDetails.length; i++) {
            //make sure we do not load two extensions with same ID
            var extensionID = extensionsDetails[i].id;
            var extensionType = extensionsDetails[i].type;
            if (this.getExtensionIndexById(extensionID) === -1) {
                //create an iframe for each extension and set its source
                var src = extensionsDetails[i].src;
                var extensionIFrame = document.createElement("iframe");
                extensionIFrame.src = this.liveExtensionsConfigPath + src;
                extensionIFrame.style.cssText = iFrameCSS;
                extensionIFrame.id = extensionID;
                //add identifying attribute for extension iframes
                extensionIFrame.setAttribute("dwExtensionFrame", "true");
                extensionIFrame.setAttribute("random", Math.random());
                if (extensionType === "auxiliary") {
                    var tags =  extensionsDetails[i].tags;
                    if (tags) {
                        if (tags.indexOf(" ") !== -1) {
                            tags = tags.split(" ");
                        }
                        if (typeof tags === "string") {
                            this.auxiliaryIframeExists[tags] = true;
                        } else if (typeof tags === "object") {
                            var j;
                            for (j = 0; j < tags.length; j++) {
                                var currentTag = tags[j].toLowerCase();
                                this.auxiliaryIframeExists[currentTag] = true;
                            }
                        }
                    }
                }
                //add the iframe inside the container        
                extensionContainer.appendChild(extensionIFrame);
                extensionIFrame = document.getElementById(extensionID);
                this.liveViewExtensionsArray.push({id: extensionID, frame: extensionIFrame, enabled: true});

                //attach dwObject to the extension iframe
                if (extensionIFrame.contentWindow) {
                    extensionIFrame.contentWindow.liveViewObject = this;
                    extensionIFrame.contentWindow.dwObject = this.dwObject;
                    extensionIFrame.contentWindow.liveExtensionsConfigPath = this.liveExtensionsConfigPath;
					extensionIFrame.contentWindow.isExtensionInitialized = true;
                }
            }
        }
    };

    /*
    function:messageToExtensions - send a message to all extensions

    Arguments: messageDetails: type(string), params(string)

    Return: none
    */

    LiveViewObjectForExtensions.prototype.messageToExtensions = function (messageDetails) {
        var i;
        messageDetails.dwExtensionEvent = true;
        for (i = 0; i < this.liveViewExtensionsArray.length; i++) {
            if (this.liveViewExtensionsArray[i] && this.liveViewExtensionsArray[i].enabled === true) {
                if (this.liveViewExtensionsArray[i].frame && this.liveViewExtensionsArray[i].frame.contentWindow) {
                    this.liveViewExtensionsArray[i].frame.contentWindow.postMessage(messageDetails, "*");
                }
            }
        }
    };
	
	/*
    function:reInitializeLiveViewExtensions - send a message to all extensions

    Arguments: reInitialize extensions if needed

    Return: none
    */

    LiveViewObjectForExtensions.prototype.reInitializeLiveViewExtensions = function () {
        var i;
		for (i = 0; i < this.liveViewExtensionsArray.length; i++) {
            if (this.liveViewExtensionsArray[i]) {
				var extensionIFrame = this.liveViewExtensionsArray[i].frame;
                if (extensionIFrame && extensionIFrame.contentWindow) {
                    if (!extensionIFrame.contentWindow.isExtensionInitialized) {
						extensionIFrame.contentWindow.liveViewObject = this;
						extensionIFrame.contentWindow.dwObject = this.dwObject;
						extensionIFrame.contentWindow.liveExtensionsConfigPath = this.liveExtensionsConfigPath;
						extensionIFrame.contentWindow.isExtensionInitialized = true;
					}
                }
            }
        }
    };
    
    /*
    function:initializeLiveViewExtensionObjects - function to initialize the objects exposed by our infrastructure
                                                  to the passed extension frame
    Arguments: id of extension

    Return: none
    */

    LiveViewObjectForExtensions.prototype.initializeLiveViewExtensionObjects = function (extensionID) {
        var extensionIFrame = this.getExtensionIFrameById(extensionID);
        if (extensionIFrame && extensionIFrame.contentWindow) {
            extensionIFrame.contentWindow.liveViewObject = this;
            extensionIFrame.contentWindow.dwObject = this.dwObject;
            extensionIFrame.contentWindow.liveExtensionsConfigPath = this.liveExtensionsConfigPath;
            extensionIFrame.contentWindow.isExtensionInitialized = true;
        }
    };

    /*
    function:initLiveViewExtensionsInfrastructure - global function to set up the infrastructure and load
                                                    the extensions
    Arguments: none

    Return: none
    */
    var initLiveViewExtensionsInfrastructure = function () {

        /* Utility functions */

        /*
        function:updateSelectedElement - Find the element base don arguments 
                                         and set it as current selection for extensions.    
        Arguments: uniqNodeId - unique node id of the element
                   selectorString - if the node id is not present, then use selector
        Return: none
        */
        var updateSelectedElement =  function (uniqNodeId, selectorString, elementIsNonEditable) {
            var targetElement = null;
            var querySelectorString = "";
            if (uniqNodeId && uniqNodeId.length > 0) {
                //in case we have a dw node id, then use query selector to get the element with attribute
                querySelectorString = '[' + DW_LIVEEDIT_CONSTANTS.DWUniqueId + '="' + uniqNodeId + '"]';
            } else if (selectorString && selectorString.length > 0) {
                querySelectorString = selectorString;
            }
            if (querySelectorString.length > 0) {
                try {
                    var elems = document.querySelectorAll(querySelectorString);
                    if (elems && elems.length > 0) {
                        targetElement = elems[0];
                    }
                } catch (e) {
                }
            }

            window.liveViewExtensionsObject.setCurrentSelectedElement(targetElement, elementIsNonEditable);
        };

        /*
        function:DWEventToLiveView - This is the function DW calls (using evalJavascript)
                                     to send messages to LiveView

        Arguments: eventName - type of event (complete set of events @ Constants.js:DW_EVENTS)    
                   params - parameters for the event if any
        Return: none
        */
        window.DWEventToLiveView = function (eventName, params) {
            var dwEvent = {};
            dwEvent.type = eventName;
            if (params) {
                dwEvent.params = params;
            }
            //update live view selection if it is a selecion change
            if (eventName === DW_EVENTS.SelectionChanged) {
                if (params) {
                    var paramsArray = params.split(',');
                    if (paramsArray.length === 3) {
                        updateSelectedElement(paramsArray[0], paramsArray[1], paramsArray[2]);
                    }
                }
            }
			
			if (eventName === DW_EVENTS.ShowHelp) {
				var argObj = {};
				// check if text hud is visible
				if (window.globalController && window.globalController.m_txtHud && window.globalController.m_txtHud.getVisibility()) {
					argObj.id = window.globalController.m_txtHud.HelpId;
					window.DWLECallJsBridgingFunction(window.globalController.m_txtHud.getHudName(), "DWSMLaunchHelp", argObj, false);
                } else {	// check if other extension huds are visible
					var liveViewExtensionObject = window.liveViewExtensionsObject;
					var len = (liveViewExtensionObject) ? (liveViewExtensionObject.liveViewExtensionsArrayForHelp.length) : 0,
                        i = 0;
					for (i = 0; i < len; i++) {
						var extensionID = liveViewExtensionObject.liveViewExtensionsArrayForHelp[i].extensionId;
						if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.isExtensionVisible(extensionID)) {
							argObj.id = liveViewExtensionObject.liveViewExtensionsArrayForHelp[i].helpId;
							window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, "DWSMLaunchHelp", argObj, false);
							break;
						}
					}
				}
			}
			
			if (eventName === DW_EXTENSION_EVENT.HIDE_AUX_HUD) {
                if (typeof params !== "undefined") {
                    if (params === "true") {
                        dwEvent.commit = true;
                    }
                }
            }

            //pass the event to extensions
            window.liveViewExtensionsObject.messageToExtensions(dwEvent);

            //should the extensions be enabled/disabled?
            if (eventName === DW_EVENTS.DisableExtensions) {
                window.liveViewExtensionsObject.disableAllExtensions();
            } else if (eventName === DW_EVENTS.EnableExtensions) {
                window.liveViewExtensionsObject.enableAllExtensions();
                
                //send a selection changed event to notify extensions
                var dwSelectionEvent = {};
                dwSelectionEvent.type = DW_EVENTS.SelectionChanged;
                window.liveViewExtensionsObject.messageToExtensions(dwSelectionEvent);
            }
        };

        window.liveViewExtensionsObject = new LiveViewObjectForExtensions();
        window.liveViewExtensionsObject.loadAllExtensions();
              
        /*
        function:shouldExtensionsHandleUndoRedo - Check whether UndoRedo commands should go to the Frame

        Arguments: none
        
        Return: true if the focus is in any of the editable areas in frame, false otherwise
        */
        window.shouldExtensionsHandleUndoRedo = function () {
            var activeElement = document.activeElement;
            if (activeElement && activeElement.tagName === "IFRAME") {
                if (activeElement.id && window.liveViewExtensionsObject.isExtensionVisible(activeElement.id)) {
                    var activeElementInFrame = activeElement.contentDocument ? activeElement.contentDocument.activeElement : null;
                    if (activeElementInFrame && (activeElementInFrame.tagName === "INPUT"
                                                 || activeElementInFrame.tagName === "TEXTAREA"
                                                 || activeElementInFrame.getAttribute("contenteditable") === "true")) {
                        return true;
                    }
                }
            }
            return false;
        };

        var messageHandlerForLiveEditExtensions = function (evt) {
            if (evt.data.dwExtensionEvent === true) {
                var property;
                for (property in DW_EXTENSION_EVENT) {
                    if (DW_EXTENSION_EVENT[property] === evt.data.type) {
                        window.liveViewExtensionsObject.messageToExtensions(evt.data);
                    }
                }
            }
        };
        
        window.addEventListener("message", messageHandlerForLiveEditExtensions, false);
		
		window.reInitializeLiveViewExtensions = function () {
			window.liveViewExtensionsObject.reInitializeLiveViewExtensions();
		};
    };

    initLiveViewExtensionsInfrastructure();
}());
//@ sourceURL=InitLiveViewExtensionsInfrastructure.js
